<?php
header("Content-Type: application/json"); // Set response header for JSON output
require('config.php'); // Database configuration file

// Check if customerData is set in POST
if (!isset($_POST["customerData"])) {
    echo json_encode(array("error" => true, "message" => "No customer data provided."));
    exit;
}

// Decode the JSON data from the POST field
$customerData = json_decode($_POST["customerData"], true);

// Validate the decoded JSON data
if (!$customerData || !is_array($customerData) || count($customerData) === 0) {
    echo json_encode(array("error" => true, "message" => "Invalid customer data format."));
    exit;
}

// Get the first customer data entry
$customer = $customerData[0];
$accountNumber = $customer['account_number'];

// Check if account_number is provided
if (empty($accountNumber)) {
    echo json_encode(array("error" => true, "message" => "Account number is required for updating customer data."));
    exit;
}

// Prepare the SQL query dynamically, skipping empty fields
$sql = "UPDATE customer_data SET ";
$fieldsToUpdate = array();
$params = array();

// Loop through each field in the customer data
foreach ($customer as $field => $value) {
    // Exclude empty fields, 'id', and 'account_number'
    if (!empty($value) && $field !== "id" && $field !== "account_number") {
        $fieldsToUpdate[] = "$field = ?";
        $params[] = $value;
    }
}

// If there are fields to update, proceed with SQL execution
if (count($fieldsToUpdate) > 0) {
    // Construct the SQL statement
    $sql .= implode(", ", $fieldsToUpdate) . " WHERE account_number = ?";
    $params[] = $accountNumber;

    // Prepare the SQL statement
    $stmt = $connect->prepare($sql);
    if ($stmt) {
        // Bind parameters dynamically
        $stmt->bind_param(str_repeat("s", count($params)), ...$params);

        // Execute the update query
        if ($stmt->execute()) {
            echo json_encode(array("error" => false, "message" => "Customer data updated successfully"));
        } else {
            echo json_encode(array("error" => true, "message" => "Error updating customer data: " . $stmt->error));
        }

        // Close the statement
        $stmt->close();
    } else {
        echo json_encode(array("error" => true, "message" => "Error preparing update statement: " . $connect->error));
    }
} else {
    echo json_encode(array("error" => true, "message" => "No fields to update."));
}

// Close the database connection
$connect->close();
?>
